-- search_rag_content: generic RAG search over rag_content with metadata filters
--
-- This function is exposed via Supabase RPC and called by the FastAPI /search endpoint.
-- It uses pgvector cosine distance on rag_content.embedding and applies AND-style
-- filters across various metadata fields.

create or replace function search_rag_content(
  query_embedding        vector(1536),

  author_prm             text default null,
  language_prm           text default null,
  isbn_prm               text default null,
  book_id_prm            text default null,
  genre_prm              text default null,
  tags_all_prm           text[] default null,
  tags_any_prm           text[] default null,
  published_from_prm     date default null,
  published_to_prm       date default null,
  chapter_prm            text default null,
  section_prm            text default null,
  edition_prm            text default null,
  source_path_prm        text default null,
  source_id_prm          text default null,

  min_score_prm          double precision default null,
  match_count            int default 10
)
returns table (
  id               bigint,
  content          text,
  title            text,
  author           text,
  isbn             text,
  book_id          text,
  chapter          text,
  chapter_num      integer,
  section          text,
  language         text,
  genre            text,
  tags             text[],
  edition          text,
  published_date   date,
  source_path      text,
  source_id        text,
  score            double precision
)
language sql
stable
as $$
  select
    r.id,
    r.content,
    r.title,
    r.author,
    r.isbn,
    r.book_id,
    r.chapter,
    r.chapter_num,
    r.section,
    r.language,
    r.genre,
    r.tags,
    r.edition,
    r.published_date,
    r.source_path,
    r.source_id,
    1 - (r.embedding <=> query_embedding) as score
  from rag_content r
  where
    -- (optional) restrict to book content; drop this line if you want all docs
    (r.document_type is null or r.document_type = 'book_paragraph')

    -- Author
    and (
      author_prm is null
      or trim(lower(r.author)) = trim(lower(author_prm))
    )

    -- Language
    and (
      language_prm is null
      or r.language = language_prm
    )

    -- ISBN / book_id
    and (
      isbn_prm is null
      or r.isbn = isbn_prm
    )
    and (
      book_id_prm is null
      or r.book_id = book_id_prm
    )

    -- Genre
    and (
      genre_prm is null
      or r.genre = genre_prm
    )

    -- Tags: ALL
    and (
      tags_all_prm is null
      or r.tags @> tags_all_prm
    )

    -- Tags: ANY
    and (
      tags_any_prm is null
      or r.tags && tags_any_prm
    )

    -- Published date range
    and (
      published_from_prm is null
      or r.published_date >= published_from_prm
    )
    and (
      published_to_prm is null
      or r.published_date <= published_to_prm
    )

    -- Chapter / section
    and (
      chapter_prm is null
      or trim(lower(r.chapter)) = trim(lower(chapter_prm))
    )
    and (
      section_prm is null
      or trim(lower(r.section)) = trim(lower(section_prm))
    )

    -- Edition
    and (
      edition_prm is null
      or r.edition = edition_prm
    )

    -- Source path / id
    and (
      source_path_prm is null
      or r.source_path = source_path_prm
    )
    and (
      source_id_prm is null
      or r.source_id = source_id_prm
    )

    -- Optional min_score filter (score = 1 - cosine_distance)
    and (
      min_score_prm is null
      or 1 - (r.embedding <=> query_embedding) >= min_score_prm
    )

  order by r.embedding <=> query_embedding
  limit match_count;
$$;
