create or replace function match_book_queries(
  query_embedding        vector(1536),
  book_title_prm         text default null,
  chapter_title_prm      text default null,
  author_prm             text default null,
  language_prm           text default null,
  isbn_prm               text default null,
  book_id_prm            text default null,
  genre_prm              text default null,
  tags_all_prm           text[] default null,
  tags_any_prm           text[] default null,
  published_from_prm     date default null,
  published_to_prm       date default null,
  edition_prm            text default null,
  source_path_prm        text default null,
  source_id_prm          text default null,
  min_score_prm          double precision default null,
  match_count            int default 10
)
returns table (
  id               bigint,
  content          text,
  title            text,
  author           text,
  isbn             text,
  book_id          text,
  chapter          text,
  chapter_num      integer,
  language         text,
  genre            text,
  tags             text[],
  edition          text,
  published_date   date,
  source_path      text,
  source_id        text,
  score            double precision
)
language sql stable
as $$
  select
    r.id,
    r.content,
    r.title,
    r.author,
    r.isbn,
    r.book_id,
    r.chapter,
    r.chapter_num,
    r.language,
    r.genre,
    r.tags,
    r.edition,
    r.published_date,
    r.source_path,
    r.source_id,
    1 - (r.embedding <=> query_embedding) as score
  from rag_content r
  where r.document_type = 'book_paragraph'

    -- Book title filter
    and (
      book_title_prm is null
      or trim(lower(r.title)) = trim(lower(book_title_prm))
    )

    -- Chapter title filter
    and (
      chapter_title_prm is null
      or trim(lower(r.chapter)) = trim(lower(chapter_title_prm))
    )

    -- Author
    and (
      author_prm is null
      or trim(lower(r.author)) = trim(lower(author_prm))
    )

    -- Language
    and (
      language_prm is null
      or r.language = language_prm
    )

    -- ISBN or book_id
    and (
      isbn_prm is null
      or r.isbn = isbn_prm
    )
    and (
      book_id_prm is null
      or r.book_id = book_id_prm
    )

    -- Genre
    and (
      genre_prm is null
      or r.genre = genre_prm
    )

    -- Tags: ALL
    and (
      tags_all_prm is null
      or r.tags @> tags_all_prm
    )

    -- Tags: ANY
    and (
      tags_any_prm is null
      or r.tags && tags_any_prm
    )

    -- Published date range
    and (
      published_from_prm is null
      or r.published_date >= published_from_prm
    )
    and (
      published_to_prm is null
      or r.published_date <= published_to_prm
    )

    -- Edition
    and (
      edition_prm is null
      or r.edition = edition_prm
    )

    -- Source path / id
    and (
      source_path_prm is null
      or r.source_path = source_path_prm
    )
    and (
      source_id_prm is null
      or r.source_id = source_id_prm
    )

    -- Optional min_score
    and (
      min_score_prm is null
      or 1 - (r.embedding <=> query_embedding) >= min_score_prm
    )

  order by r.embedding <=> query_embedding
  limit match_count;
$$;
